function extractMenuItems() {
  const items = [];
  const itemElements = document.querySelectorAll(
    '[class*="menuItem"], [class*="MenuItem"], [class*="item-card"], [data-testid*="menu-item"], [class*="itemCard"], .menu-item, .menuItem, [class*="MenuItemCard"]'
  );
  for (const el of itemElements) {
    const item = parseToastItem(el);
    if (item) items.push(item);
  }
  if (items.length === 0) {
    const allElements = document.querySelectorAll("div, li, article, section");
    for (const el of allElements) {
      if (el.children.length > 10) continue;
      const text = el.textContent ?? "";
      const priceMatch = text.match(/\$(\d+\.\d{2})/);
      if (!priceMatch) continue;
      const nameEl = el.querySelector('h3, h4, h5, [class*="name"], [class*="title"], span:first-child');
      const name = nameEl?.textContent?.trim();
      if (!name || name.includes("$")) continue;
      const price = Math.round(parseFloat(priceMatch[1]) * 100);
      if (price > 0 && price < 5e4) {
        items.push({ name, price });
      }
    }
  }
  const ldScripts = document.querySelectorAll('script[type="application/ld+json"]');
  for (const script of ldScripts) {
    try {
      const data = JSON.parse(script.textContent ?? "");
      if (data["@type"] === "Menu" || data["@type"] === "Restaurant") {
        const menuItems = data.hasMenu?.hasMenuSection?.flatMap(
          (s) => s.hasMenuItem ?? []
        ) ?? [];
        for (const mi of menuItems) {
          if (mi.name && mi.offers?.price) {
            items.push({
              name: mi.name,
              price: Math.round(parseFloat(mi.offers.price) * 100),
              description: mi.description
            });
          }
        }
      }
    } catch {
    }
  }
  const nextData = document.getElementById("__NEXT_DATA__");
  if (nextData) {
    try {
      const data = JSON.parse(nextData.textContent ?? "");
      extractFromNestedObject(data, items);
    } catch {
    }
  }
  return deduplicateItems(items);
}
function parseToastItem(el) {
  const nameEl = el.querySelector(
    '[class*="itemName"], [class*="name"], [class*="title"], h3, h4, h5'
  );
  const priceEl = el.querySelector(
    '[class*="price"], [class*="Price"], [class*="cost"]'
  );
  const name = nameEl?.textContent?.trim();
  const priceText = priceEl?.textContent?.trim();
  if (!name || !priceText) return null;
  const priceMatch = priceText.match(/\$?(\d+\.\d{2})/);
  if (!priceMatch) return null;
  const price = Math.round(parseFloat(priceMatch[1]) * 100);
  if (price <= 0 || price > 5e4) return null;
  const descEl = el.querySelector(
    '[class*="description"], [class*="desc"], p'
  );
  return {
    name,
    price,
    description: descEl?.textContent?.trim()
  };
}
function extractFromNestedObject(obj, items, depth = 0) {
  if (depth > 10 || !obj || typeof obj !== "object") return;
  if (obj.name && (obj.price || obj.amount)) {
    const price = typeof obj.price === "number" ? obj.price : typeof obj.amount === "number" ? obj.amount : parseFloat(String(obj.price || obj.amount));
    if (!isNaN(price) && price > 0 && price < 500) {
      items.push({
        name: String(obj.name),
        price: Math.round(price * 100),
        description: obj.description ? String(obj.description) : void 0
      });
    }
  }
  for (const key of Object.keys(obj)) {
    if (Array.isArray(obj[key])) {
      for (const item of obj[key]) {
        extractFromNestedObject(item, items, depth + 1);
      }
    } else if (typeof obj[key] === "object") {
      extractFromNestedObject(obj[key], items, depth + 1);
    }
  }
}
function deduplicateItems(items) {
  const seen = /* @__PURE__ */ new Set();
  return items.filter((item) => {
    const key = `${item.name.toLowerCase()}:${item.price}`;
    if (seen.has(key)) return false;
    seen.add(key);
    return true;
  });
}
function init() {
  setTimeout(() => {
    const items = extractMenuItems();
    if (items.length > 0) {
      chrome.runtime.sendMessage({
        type: "TOAST_MENU_SCRAPED",
        payload: {
          url: window.location.href,
          restaurant: document.title.replace(/\s*[-|].*$/, "").trim(),
          items,
          scrapedAt: (/* @__PURE__ */ new Date()).toISOString()
        }
      });
    }
  }, 3e3);
}
let lastUrl = window.location.href;
const observer = new MutationObserver(() => {
  if (window.location.href !== lastUrl) {
    lastUrl = window.location.href;
    setTimeout(() => init(), 2e3);
  }
});
observer.observe(document.body, { childList: true, subtree: true });
init();
